//---------------------------------------------------------------------------------------------
//
//	Name:	Best64.c
//	Create:	3/2/98
//	Rev:	1.00
//	Date:	5/15/98
//	Desc:	The E2926A system tests
//	Editor:	Jim Carlisi
//	Problems and fixes:		3/05/98	Changed all C-API's to E2926A compliant
//							3/05/98	Added better use of "threads"
//							3/05/98	Log file appends instead of starting a new one
//							3/19/98 Open_all_cards checks for E2925 or E2926
//							3/30/98 Added .RC file with version number
//							4/10/98 Changed all addresses to LARGE_INTEGERS for 64bit support
//							5/15/98 Set PPR to try 32 and 64 bit cycles
//							5/15/98 Removed some TODOs
//							5/15/98 Inserted cpu_mem and cpu_io test with write, address, read data output
//							5/15/98 
//---------------------------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <windows.h>
#include <winnt.h>
#include <conio.h>
#include <time.h>				//CBB 8/4/97

#include <mini_api.h>
#include <ppr.h>
#include <b_ntmem.h>

#define C(x) if ((err = (x)) != B_E_OK) \
{ printf("%s (line %d)\n", BestErrorStringGet(err), __LINE__);\
}
  
#define HP_VEN_ID		0x103c		// This is the HP Ven ID and is the same for E2925 and E2926
#define E2925A_DEV_ID	0x2925		// The 2925 device ID
#define E2926A_DEV_ID	0x2926		// The 2926 device ID

#define STATUS_MASTER_ACTIVE            0x01
#define STATUS_PROTOCOL_ERROR           0x10
#define STATUS_DATA_COMPARE_ERROR       0x20

#define BEST_32_BIT_CARD		0			// 32 bit card Identifier
#define BEST_64_BIT_CARD		1			// 64 bit card Identifier

#define MY_ATTR_PAGE    2
#define MAX_CARDS       32					// from MaxHandles in typedefs.h (keep values the same)
#define MAX_THREADS     32					// from MaxHandles in typedefs.h (keep values the same)

#define WRITE_PPR_REP    1
#define NO_PPR_REP       0

typedef struct card_info_s 
{	struct thread_info_s *tip;	// points to the thread that owns this card
	
	int bitver;					// 0=32 bit card, 1=64 bit card
	int busy;					// indicates whether the device is busy
	b_int32 devid;				// device ID returned by BestDevIdentifierGet
	b_handletype handle;		// handle from BestOpen

	LARGE_INTEGER memaddr;		// memory base address of on-board memory (physical) (memaddr.LowPart, memaddr.HiPart)
	b_int32 memsize;		    // size of memory
	void *memptr;				// pointer to access on-board memory (virtual addr.)

	LARGE_INTEGER ioaddr;		// I/O base address (physical) (ioaddr.LowPart, ioaddr.HighPart)
	b_int32 iosize;				// size of I/O space
	void *ioptr;				// pointer to access on-board I/O space (virtual addr.)

	void *sysmem_ptr;			// pointer to allocated system memory
	LARGE_INTEGER sysmem_addr;	// physical address of allocated system memory


} card_info_t;

card_info_t card_info[MAX_CARDS];   // array with information about all cards
unsigned int n_cards = 0;           // actual number of cards located in the system

typedef enum { STARTING, RUNNING, TERMINATED } thread_state_t;

typedef struct thread_info_s 
{	DWORD idThread;				// ID of the thread
	HANDLE hThread;				// handle of the thread
	thread_state_t state;		// the current state of the thread
	int term_request;			// =1, request to terminate
	double maxtime;				// runtime for this thread
	struct fct_tab_s *ftp;		// pointer to test function that this thread executes
} thread_info_t;

thread_info_t thread_info[MAX_THREADS];		// Array with information about all threads
unsigned int n_threads = 0;					// Thread counter - keeps track of thread use

typedef struct fct_tab_s 
{	char *name;						// name of the function
	LPTHREAD_START_ROUTINE fct;		// pointer to function
} fct_tab_t;

extern fct_tab_t fct_tab[];			// declaration for test function table

//---------------------------------------------------------------------------------------------
// functions from physical.c
//---------------------------------------------------------------------------------------------

int allocate_physical_memory(unsigned long, void **, unsigned long *);
int free_physical_memory(void *, unsigned long);
void *map_physical_address(unsigned long phys_addr_low, unsigned long phys_addr_hi, unsigned long length,
			   unsigned int bus_number, unsigned int io);
void unmap_physical_address(void *virt_addr);
int io_access(int cmd, unsigned long addr, unsigned long *data);
//int test_physical_mem();
//unsigned long virtual_to_physical(unsigned long addr);

//---------------------------------------------------------------------------------------------
// global variables
//---------------------------------------------------------------------------------------------

void *aptr = 0;                             // pointer to allocated virt. memory (0=not allocated)
unsigned long phys_addr_low;				// physical address of allocated memory (lower)
unsigned long phys_addr_hi;					// physical address of allocated memory (hi)
const b_int32 page_size = 4096;             // size of physical memory pages
unsigned long cacheline_size = 0;           // system's cachelinesize, will be set by open_all_cards
int use_ppr = 1;                            // 1=using PPR variations 0=no variations
int write_report = 1;                       // flag for generating a PPR report
int dbg = 1;								// dbg=0: no output, =1: verbose output
FILE *stream;								// CBB 8/4/97
double default_maxtime = 10.0;				// default runtime for tests


//---------------------------------------------------------------------------------------------
// Set up block permutations to be used when reading or writing data.
// This routine assumes, that a connection to an E2926A is already established and
// that BestPprInit has been called already
//---------------------------------------------------------------------------------------------

b_errtype
setup_master_block(b_handletype handle, 
				   b_int32 dir, 
				   LARGE_INTEGER mem_addr, 
				   b_int32 int_addr,
				   b_int32 size, 
				   b_int32 bpage, 
				   b_int32 compflag, 
				   b_int32 compoffs, 
				   int is64bitcard,
				   int writePPRreport,
				   b_int32 lineno,
				   int is_last_line)
{	b_errtype err;
	b_int32 val;
	char align_str[50];
	bppr_algorithmtype alg = BPPR_ALG_PERM;
	struct tm when;
	time_t now;
	

	time(&now);
	when = *localtime(&now);

	if (use_ppr)
	{
		C(BestPprInit(handle));
		//----- set up PPR block properties (writing mem_size bytes to mem_addr)
		C(BestPprBlockInit(handle));												// Initializes the block permutator
		C(BestPprBlockPermPropDefaultSet(handle));									// Sets the block permutator to default properties
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_DIR,         dir));				// Sets a block permutation property
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_BUSADDR,     mem_addr.LowPart));	
		if (is64bitcard == BEST_64_BIT_CARD)
		{	C(BestPprBlockPermPropSet(handle, BPPR_BLK_BUSADDR_HI,  mem_addr.HighPart));
		}
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_INTADDR,     int_addr));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_NOFDWORDS,   size / 4));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_ATTRPAGE,    MY_ATTR_PAGE));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_PAGENUM,     bpage));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_PAGESIZEMAX, 32));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_FILLGAPS,    1));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_CACHELINE,   cacheline_size));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_COMPFLAG,    compflag));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_COMPOFFS,    compoffs));

		//----- set up PPR block variations
		//----- should be modified according to system requirements
		C(BestPprBlockVariationDefaultSet(handle));									// Sets block variation parameters to default sizes
		C(BestPprBlockVariationSet(handle, BPPR_BLK_SIZE,  "4,8,16,32", alg));		// Defines a value list for a block variation parameter

		// some normal and some "odd" byte enables
		C(BestPprBlockVariationSet(handle, BPPR_BLK_BYTEN, "10,0,9,3,11", alg));

		// all read and commands (read, read multiple, read line, write, write & invalidate)
		C(BestPprBlockVariationSet(handle, BPPR_BLK_CMDS,  "6,12,14,7", alg));

		// if no cache line size is set, assume it is 8
		val = cacheline_size;
		if (val == 0)
		{	val = 8;
		}

		sprintf(align_str, "(%%%ld=0),(%%%ld=4),(%%%ld=%ld)", 4*val, 4*val, 4*val, 4*val-4);
		
		//printf("Alignment varitions: %s\n", align_str);
		C(BestPprBlockVariationSet(handle, BPPR_BLK_ALIGN, align_str, alg));		// An algorithm for permutations

		if (writePPRreport)
		{	sprintf(align_str, "\\ppr%02d.rpt", write_report);
			printf("Generating PPR report...(%s)\n", align_str);
			fprintf(stream, "%s Generating PPR report...(%s)\n", asctime (&when), align_str);   //CBB,8/4/97
			C(BestPprReportPropDefaultSet(handle));
			C(BestPprReportFile(handle, align_str));                                             //CBB,8/4/97
			write_report++;
		}

		C(BestPprBlockGenerate(handle));			// Generates a compound block
		C(BestPprDelete(handle));
	}

	else
	{	
		if (bpage != BPPR_BLK_APPEND)
		{	C(BestMasterBlockPageInit(handle, bpage));
		}
		C(BestMasterBlockPropDefaultSet(handle));
		C(BestMasterBlockPropSet(handle, B_BLK_BUSADDR, mem_addr.LowPart));			// 32 bit address width
		C(BestMasterBlockPropSet(handle, B_BLK_BUSCMD, dir == BPPR_DIR_WRITE ? 7 : 6));
		C(BestMasterBlockPropSet(handle, B_BLK_BYTEN, 0));							// 0 is the default
		C(BestMasterBlockPropSet(handle, B_BLK_INTADDR, int_addr));
		C(BestMasterBlockPropSet(handle, B_BLK_NOFDWORDS, size / 4));
		C(BestMasterBlockPropSet(handle, B_BLK_ATTRPAGE, MY_ATTR_PAGE));			// 0-63 : pointer to master attrib page
		C(BestMasterBlockPropSet(handle, B_BLK_COMPFLAG, compflag));				// 0=no compare, 1=data compare
		C(BestMasterBlockPropSet(handle, B_BLK_COMPOFFS, compoffs));
		if (is64bitcard == BEST_64_BIT_CARD)
		{	
			if(mem_addr.HighPart)
				C(BestMasterBlockPropSet(handle, B_BLK_BUSDAC, 1));							// Dual Address cycle with 64 bit address
			C(BestMasterBlockPropSet(handle, B_BLK_BUSADDR_HI, mem_addr.HighPart));		// 64 bit address width

			// if you are using the new commands you must make sure that the
			// bpage is not set to B_APPEND?? or similar, bpage must always
			// contain a valid number!!!!
//			C(BestMasterBlockLineProg(handle, bpage, lineno));								// BestMasterBlockProg() replaced
//		    if(is_last_line)
//				C(BestMasterBlockEndProg(handle, bpage, lineno));							// BestMasterBlockPage() replaced
		}
//		else
			C(BestMasterBlockProg(handle));													// Used in 2925A but not in 2926A
		
	}

	return(B_E_OK);
}

//---------------------------------------------------------------------------------------------
// Set up master attribute permutations to be used when reading or writing data.
// This routine assumes, that a connection to an E2926A is already established and
// that BestPprInit has been called already
//---------------------------------------------------------------------------------------------

b_errtype
setup_master_attributes(b_handletype handle,int is64bitcard)
{	b_errtype err;
	bppr_algorithmtype alg = BPPR_ALG_PERM;
	
	if (use_ppr)
	{
		C(BestPprInit(handle));
		//----- set up PPR master attribute properties
		C(BestPprMAttrInit(handle));
		C(BestPprMAttrPermPropDefaultSet(handle));

		//----- in the program, only one attribute page is used
		C(BestPprMAttrPermPropSet(handle, BPPR_MA_PAGENUM, MY_ATTR_PAGE));
		C(BestPprMAttrPermPropSet(handle, BPPR_MA_PAGESIZEMAX, 8000));
		C(BestPprMAttrPermPropSet(handle, BPPR_MA_FIRSTPERM, 1));

		//----- set up PPR master attribute variations
		//----- should be modified according to system requirements
		C(BestPprMAttrVariationDefaultSet(handle));
		C(BestPprMAttrVariationSet(handle, B_M_LAST,    "1,2,4,8", alg));
		C(BestPprMAttrVariationSet(handle, B_M_WAITS,   "0,1,2,5,7", alg));
		if (is64bitcard == BEST_64_BIT_CARD)
		{	
			C(BestPprMAttrVariationSet(handle, B_M_REQ64,   "1,1,0,1,0", alg));
			C(BestPprMAttrVariationSet(handle, B_M_RELREQ,  "0", alg));
		}
		C(BestPprMAttrVariationSet(handle, B_M_DPERR,   "0", alg));
		C(BestPprMAttrVariationSet(handle, B_M_DSERR,   "0", alg));
		C(BestPprMAttrVariationSet(handle, B_M_APERR,   "0", alg));
		C(BestPprMAttrVariationSet(handle, B_M_DWRPAR,  "0", alg));
		C(BestPprMAttrVariationSet(handle, B_M_AWRPAR,  "0", alg));
		C(BestPprMAttrVariationSet(handle, B_M_WAITMODE,"0", alg));
		C(BestPprMAttrVariationSet(handle, B_M_STEPMODE,"0", alg));

		//----- downloading PPR attribute page
		C(BestPprMAttrGenerate(handle));
		C(BestPprDelete(handle));
	}

	else
	{
		//----- set up a simple attribute page with zero waits
		//----- bursts as long as needed
		C(BestMasterAttrPageInit(handle, MY_ATTR_PAGE));
		C(BestMasterAttrPropDefaultSet(handle));
		C(BestMasterAttrPropSet(handle, B_M_WAITS, 0));			// Default
		C(BestMasterAttrPropSet(handle, B_M_LAST, 0));			// Default
		C(BestMasterAttrPropSet(handle, B_M_DOLOOP, 1));
		if (is64bitcard == BEST_64_BIT_CARD)
		{
			C(BestMasterAttrPropSet(handle, B_M_REQ64, 1));			// 64 bit access will be tried
			C(BestMasterAttrPropSet(handle, B_M_TRYBACK, 1));		// Try fast back-to-back cycle
//			C(BestMasterAttrLineProg(handle, MY_ATTR_PAGE, 0));	// Changed in 2926A
		}
//		else
//		{	
		C(BestMasterAttrPhaseProg(handle));					// Removed for 2926A
//		}
	}
	return(B_E_OK);
}

//---------------------------------------------------------------------------------------------
// Set up target attribute permutations to be used when reading or writing data.
// This routine assumes, that a connection to an E2926A is already established and
// that BestPprInit has been called already
//---------------------------------------------------------------------------------------------

b_errtype
setup_target_attributes(b_handletype handle,int is64bitcard)
{	b_errtype err;

	bppr_algorithmtype alg = BPPR_ALG_PERM;

	if (use_ppr)
	{

		C(BestPprInit(handle));


		//----- set up PPR target attribute properties
		C(BestPprTAttrInit(handle));						// Initialize the target attribute permutator
		C(BestPprTAttrPermPropDefaultSet(handle));			// Sets target attrib permutation props to default
		//----- in the program, only one attribute page is used
		C(BestPprTAttrPermPropSet(handle, BPPR_TA_PAGENUM, MY_ATTR_PAGE));
		C(BestPprTAttrPermPropSet(handle, BPPR_TA_PAGESIZEMAX, 8000));
		C(BestPprTAttrPermPropSet(handle, BPPR_TA_FIRSTPERM, 1));

		//----- set up PPR master attribute variations
		//----- should be modified according to system requirements
		C(BestPprTAttrVariationDefaultSet(handle));
		C(BestPprTAttrVariationSet(handle, B_T_WAITS,   "0,1,2,5,7", alg));
		C(BestPprTAttrVariationSet(handle, B_T_TERM,    "0,1,2", alg)); // noterm, retry, disconnect
		C(BestPprTAttrVariationSet(handle, B_T_DPERR,   "0", alg));
		C(BestPprTAttrVariationSet(handle, B_T_DSERR,   "0", alg));
		C(BestPprTAttrVariationSet(handle, B_T_APERR,   "0", alg));
		C(BestPprTAttrVariationSet(handle, B_T_WRPAR,   "0", alg));
		if (is64bitcard == BEST_64_BIT_CARD)
		{	C(BestTargetAttrPropSet(handle, B_T_ACK64, 1));				// for use with 2926A only
		}

		//----- downloading PPR attribute page
		C(BestPprTAttrGenerate(handle));
		C(BestTargetAttrPageSelect(handle, MY_ATTR_PAGE));
		C(BestPprDelete(handle));
	}

	else
	{
		//----- set up a simple attribute page with zero waits
		//----- bursts as long as needed
		C(BestTargetAttrPageInit(handle, MY_ATTR_PAGE));
		C(BestTargetAttrPropDefaultSet(handle));
		C(BestTargetAttrPropSet(handle, B_T_WAITS, 4));				// 4 clock cycles after address phase before TRDY# asserted
		C(BestTargetAttrPropSet(handle, B_T_TERM, B_TERM_NOTERM));	// Data is accepted. No termination
		C(BestTargetAttrPropSet(handle, B_T_DOLOOP, 1));
		if (is64bitcard == BEST_64_BIT_CARD)
		{	C(BestTargetAttrPropSet(handle, B_T_ACK64, 1));				// for use with 2926A only
			C(BestTargetAttrLineProg(handle, MY_ATTR_PAGE,0));			// changed for 2926A
		}
		else
		{	C(BestTargetAttrPhaseProg(handle));							// Use in 2925A and not in 2926A
		}
		C(BestTargetAttrPageSelect(handle, MY_ATTR_PAGE));
	}

	return(B_E_OK);
}

//---------------------------------------------------------------------------------------------
// check the status of a single card. If it reports a data compare or protocol error,
// print it out and return -1. Otherwise return 0
// If flag==0, print the result only if a card fails. If flag==1 print results
// unconditionally.
//---------------------------------------------------------------------------------------------
char * getobserror(b_handletype handle,int is64bitcard);


int check_card(int flag, card_info_t *cdp)
{	b_errtype err;
	b_handletype handle = cdp->handle;
	int ret = 0;
	b_int32 status;
	b_int32 i = cdp - card_info;
	struct tm when;
	time_t now;
	char * perr = NULL;


	time(&now);
	when = *localtime(&now);

	C(BestStatusRegGet(cdp->handle, &status));
	if (flag || (status & (STATUS_DATA_COMPARE_ERROR | STATUS_PROTOCOL_ERROR)))
	{	
		if (status & (STATUS_DATA_COMPARE_ERROR | STATUS_PROTOCOL_ERROR))
			ret = -1;

		perr = getobserror(cdp->handle,cdp->bitver);
		if(perr == NULL)
			perr = "";

		printf("\n***** Card #%ld: (%02lx) %s%s%s\t %s\n",
			i, status,
			(status & STATUS_MASTER_ACTIVE ? "running" : "stopped"),
			(status & STATUS_PROTOCOL_ERROR ? ", PROTOCOL_ERROR" : ""),
			(status & STATUS_DATA_COMPARE_ERROR ? ", DATA_COMPARE_ERROR" : ""),perr);

		fprintf(stream, "\n%s ***** Card #%ld: (%021x) %s%s%s\t %s\n", asctime (&when),
			i, status,
			(status & STATUS_MASTER_ACTIVE ? "running" : "stopped"),
			(status & STATUS_PROTOCOL_ERROR ? ", PROTOCOL_ERROR" : ""),
			(status & STATUS_DATA_COMPARE_ERROR ? ", DATA_COMPARE_ERROR" : ""),perr);
		C(BestStatusRegClear(cdp->handle, 0xffffffff));  //supposed clear the status reg after DCE (doesn't 6-17-98)
	}

	return(ret);
}

//---------------------------------------------------------------------------------------------
// ask the user to select a single card from the list of identified cards
//---------------------------------------------------------------------------------------------

card_info_t *
select_single_card(thread_info_t *tip, char *prompt)
{	unsigned long i;
	card_info_t *cdp;
	b_handletype handle;
	int found = 0;
	char buffer[20];
	struct tm when;
	time_t now;

	time(&now);
	when = *localtime(&now);

rep:
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{	handle = cdp->handle;
		if (!cdp->busy)
		{
			// printf("%s, Card #%d  (bus %d, dev 0x%02lx)  %s\n", asctime (&when),
			// i, (cdp->devid >> 8) & 0xff, cdp->devid & 0xff, cdp->busy ? "busy":"idle");
			found++;
		}
	}

	if (found)
	{	printf(prompt ? prompt : "Select a card: ");
	}

	else
	{	printf("No cards found\n");
		fprintf(stream, "%s No cards found.\n", asctime (&when));
		return(NULL);
	}

	gets(buffer);
	i = strtoul(buffer, 0, 0);
	if (*buffer == 'q')
	{	return(NULL);
	}

	if (i < 0 || i >= n_cards)
	{	printf("***** Invalid selection. Valid numbers are in the range: 0...%ld\n", n_cards-1);
		goto rep;
	}

	cdp = &card_info[i];
	if (cdp->busy)
	{	printf("***** That card is busy\n");
		fprintf(stream, "%s ***** That card is busy.\n", asctime (&when));
		goto rep;
	}

	cdp->busy = 1;
	cdp->tip = tip;
	return(cdp);
}

char * getobserror(b_handletype handle,int is64bitcard)
{
  b_int32 status;
  b_errtype err;
  b_int32 bitpos = 0;
  char * errstr = NULL;

  C(BestObsStatusGet(handle,B_OBS_FIRSTERR, &status));
  if(!status && is64bitcard == BEST_64_BIT_CARD)
  { // we have a E2926A here and firsterr is zero so far
    C(BestObsStatusGet(handle,B_OBS_FIRSTERR2, &status));
    bitpos=32;
  }

  if(status)
  {
      while(status >>= 1)
	      bitpos++;
	  C(BestObsErrStringGet(handle,bitpos,&errstr));
  }
  
  return errstr;
}
//---------------------------------------------------------------------------------------------
// open_all_cards searches the system for E2926A's, opens a connection to them
// and reads out the assigned memory and I/O ranges.
// after return from this function, n_cards will be set and the card_info array
// will be filled. If an error occurs during init, -1 is returned. Otherwise 0
//---------------------------------------------------------------------------------------------

int open_all_cards()
{	unsigned int count;			// subsys ID (number) for BestDevIdentifierGet()
	card_info_t *cdp;			// points to card info
	b_errtype err;
	b_handletype handle;		// handle to a filesystem
	b_int32 mode;
	int ret = 0;
	struct tm when;
	time_t now;
	int i,myport=B_PORT_COM1;
	char * fw = NULL,*cp = NULL;
	char buf[30];
	b_decodertype iodecoder;

	time(&now);
	when = *localtime(&now);

	// locate all cards in the system
	for (i=0,n_cards = 0, cdp = card_info; n_cards < MAX_CARDS; n_cards++,i++)
	{	
		err = BestDevIdentifierGet(HP_VEN_ID, E2925A_DEV_ID, i, &(cdp->devid));
		cdp->bitver = BEST_32_BIT_CARD;							// set bit version to 64 bit card
		if (err == B_E_NO_BEST_PCI_DEVICE_FOUND)
		{
				break;			// will break out of for loop
		}

		if (err != B_E_OK)
		{	printf("BestDevIdentifierGet failed: %s\n", BestErrorStringGet(err));
			fprintf(stream, "%s BestDevIdentifierGet failed: %s\n", asctime (&when), BestErrorStringGet(err));
			break;
		}

		printf("Card (E2925A) #%2d found (bus %d, slot %d, dev 0x%02lx)\n", 
			n_cards,					// card number
			(cdp->devid >> 8) & 0xff,	// bus number
			(cdp->devid >> 3) /*& 0xff*/,	// Slot number
			cdp->devid & 0xff);			// unique device ID number
		fprintf(stream, "%s Card (E2925A) #%d found (bus %d, slot %d, dev 0x%02lx)\n", 
			asctime (&when),			// Time
			n_cards,					// card number
			(cdp->devid >> 8) & 0xff,	// bus number
			(cdp->devid >> 3) & 0xff,	// slot number
			cdp->devid & 0xff);			// unique device ID

		cdp++;
	}

	for (i=0; n_cards < MAX_CARDS; n_cards++,i++)
	{	
		err = BestDevIdentifierGet(HP_VEN_ID, E2926A_DEV_ID, i, &(cdp->devid));
		cdp->bitver = BEST_64_BIT_CARD;							// set bit version to 64 bit card
		if (err == B_E_NO_BEST_PCI_DEVICE_FOUND)
		{
				break;			// will break out of for loop
		}

		if (err != B_E_OK)
		{	printf("BestDevIdentifierGet failed: %s\n", BestErrorStringGet(err));
			fprintf(stream, "%s BestDevIdentifierGet failed: %s\n", asctime (&when), BestErrorStringGet(err));
			break;
		}

		printf("Card (E2926A) #%2d found (bus %d, slot %d, dev 0x%02lx)\n", 
			n_cards,					// card number
			(cdp->devid >> 8) & 0xff,	// bus number
			(cdp->devid >> 3) /*& 0xff*/,	// Slot number
			cdp->devid & 0xff);			// unique device ID number
		fprintf(stream, "%s Card (E2926A) #%d found (bus %d, slot %d, dev 0x%02lx)\n", 
			asctime (&when),			// Time
			n_cards,					// card number
			(cdp->devid >> 8) & 0xff,	// bus number
			(cdp->devid >> 3) & 0xff,	// slot number
			cdp->devid & 0xff);			// unique device ID

		cdp++;
	}


	if (n_cards == 0)
	{	printf("no E2925A and no E2926A found\n");
		fprintf(stream, "%s No E2926A found.\n", asctime (&when));
		ret=-1;
	}

	// open connections to all the cards and read out their base addresses
	for (count = 0, cdp = card_info; count < n_cards; count++, cdp++)
	{
		// open a connection to the card
		C(BestOpen(&handle, B_PORT_PCI_CONF, cdp->devid));

		cdp->handle = handle;					// BestOpen handle = card_info handle
		cdp->busy = 0;
		cdp->tip = NULL;						// thread info handle (no thread owns)
		C(BestConnect(cdp->handle));
			// show the card number on the display (does nothing on the 64 bit card)
		C(BestDisplayPropSet(cdp->handle, B_DISP_USER));		// user mode -> allows BestDisplayWrite() function
		C(BestDisplayWrite(cdp->handle, count));				// Displays the card # -> 0-99

		C(BestVersionGet(cdp->handle,B_VER_FIRMWARE,&fw));
        buf[0] = '\0';
		strcat(buf,fw);

		C(BestVersionGet(cdp->handle,B_VER_CAPI,&cp));
				
		printf("Card #%d: %s capi: (%s) firmware (%s)\n", count,
			cdp->bitver == BEST_32_BIT_CARD ? "E2925A" : "E2926A",cp,buf);
		fprintf(stream,"Card #%d: %s capi: (%s) firmware (%s)\n", count,
			cdp->bitver == BEST_32_BIT_CARD ? "E2925A" : "E2926A",cp,buf);

		C(BestMasterStop(cdp->handle));							// just to be on the safe side: stop master
		C(BestStatusRegClear(cdp->handle, 0xffffffff));			// reset data compare errors

		C(BestObsStatusClear(cdp->handle));						// reset protocol errors
		C(BestConfRegGet(cdp->handle, 0x0c, &cacheline_size));	// Reads the value of a reg in config space (offset by 0x0c)
		cacheline_size &= 0xff;

		if (cacheline_size == 0)
		{	printf("***** Card #%d: cacheline size was not set by system BIOS\n", count);
			fprintf(stream, "%s ***** Card #%d: cacheline size was not set by system BIOS\n", asctime (&when), count);
			ret = -1;		//kicks program out to lunch and no test can be run
		}
		C(BestObsMaskSet(cdp->handle,B_R_SEM_9,1));
		C(BestObsMaskSet(cdp->handle,B_R_SEM_8,1));

		// set the master generic properties of the card to default
		C(BestMasterGenPropDefaultSet(cdp->handle));
//		C(BestMasterGenPropSet(cdp->handle, B_MGEN_LATMODE, B_LATMODE_ON));
		C(BestMasterGenPropSet(cdp->handle, B_MGEN_ATTRMODE, B_ATTRMODE_SEQUENTIAL));

		// find memory base address of the card, assume it is in decoder #1 (=BAR0)
		C(BestTargetDecoderRead(cdp->handle, B_DEC_STANDARD_1));
		if(cdp->bitver == BEST_32_BIT_CARD)			// denotes 32bit card 
		{
			C(BestTargetDecoderPropGet(cdp->handle, B_DEC_MODE, &mode));
		}
		else							// (cdp->bitver == BEST_64_BIT_CARD) 64 bit card
		{
			C(BestTargetDecoderPropGet(cdp->handle, B_DEC_LOCATION, &mode));		//changed from MODE to LOCATION
			C(BestTargetDecoderPropGet(cdp->handle, B_DEC_BASEADDR_HI, &(cdp->memaddr.HighPart)));
		}
		
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_SIZE, &(cdp->memsize)));
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_BASEADDR, &(cdp->memaddr.LowPart)));

		if(cdp->bitver == BEST_32_BIT_CARD)			//B_MODE_MEM only in 32 bit code
		{
			if (mode != B_MODE_MEM || cdp->memaddr.LowPart == 0)
			{	printf("***** Card #%d: BAR0 is not a memory base address or not assigned by the BIOS\n", count);
				fprintf(stream, "%s ***** Card #%d: BAR0 is not a memory base address or not assigned by the BIOS\n", asctime (&when), count);
				cdp->memptr = NULL;
				cdp->memaddr.LowPart = 0;
				cdp->memaddr.HighPart = 0;
				cdp->memsize = 0;
				ret = -1;
			}

			else
			{
				iodecoder = B_DEC_STANDARD_2; // look here for io space
				cdp->memaddr.HighPart = 0x0;
				cdp->memsize = 1 << cdp->memsize;
				if(cdp->memsize > 4096)
					cdp->memsize = 4096;
				cdp->memptr = (b_int8 *)map_physical_address(cdp->memaddr.LowPart, cdp->memaddr.HighPart, cdp->memsize, (cdp->devid >> 8) & 0xff, 0);
				if (cdp->memptr == NULL)
				{
					printf("***** Card #%d: can't map physical into virtual memory\n", count);
					fprintf(stream, "%s ***** Card #%d: can't map physical into virtual memory\n", asctime (&when), count);
				}

				if (dbg)
				{
					printf("Card #%d: mem base addr: 0x%08lx HI: 0x%08lx (size 0x%lx),\n\t virtual addr: 0x%08lx\n",
						count,								// card number
						cdp->memaddr.LowPart,				// lower 32 bits of address
						cdp->memaddr.HighPart,
						cdp->memsize,
						(unsigned long)cdp->memptr);
					fprintf(stream, "%s Card #%d: mem base addr: 0x%08lx HI: 0x%08lx (size 0x%1x),\n\t virtual addr: 0x%08lx\n", asctime (&when),
						count,								// card number
						cdp->memaddr.LowPart,				// lower 32 bits of address
						cdp->memaddr.HighPart,
						cdp->memsize, 
						(unsigned long)cdp->memptr);
				}
			}
		}

		else					//(cdp->bitver == BEST_64_BIT_CARD) 64 bit code only
		{
			if (/*mode != B_LOC_SPACE64 ||*/ cdp->memaddr.LowPart == 0)
			{	printf("***** Card #%d: BAR0 is not a memory base address or not assigned by the BIOS\n", count);
				fprintf(stream, "%s ***** Card #%d: BAR0 is not a memory base address or not assigned by the BIOS\n", asctime (&when), count);
				cdp->memptr = NULL;
				cdp->memaddr.LowPart = 0;
				cdp->memaddr.HighPart = 0;
				cdp->memsize = 0;
				ret = -1;
			}

			else
			{
				iodecoder = B_DEC_STANDARD_3; // look here for io space
				if(cdp->memsize > 4096)
					cdp->memsize = 4096;
				cdp->memsize = 1 << cdp->memsize;
				cdp->memptr = (b_int8 *)map_physical_address(cdp->memaddr.LowPart, cdp->memaddr.HighPart, cdp->memsize, (cdp->devid >> 8) & 0xff, 0);
				if (cdp->memptr == NULL)
				{
					printf("***** Card #%d: can't map physical into virtual memory\n", count);
					fprintf(stream, "%s ***** Card #%d: can't map physical into virtual memory\n", asctime (&when), count);
				}

				if (dbg)
				{
					printf("Card #%d: mem base addr_low: 0x%08lx addr_hi: 0x%08lx (size 0x%lx),\n\t virtual addr: 0x%08lx\n",
						count,
						cdp->memaddr.LowPart,
						cdp->memaddr.HighPart,
						cdp->memsize,
						(unsigned long)cdp->memptr);
					fprintf(stream, "%s Card #%d: mem base addr_low: 0x%08lx addr_hi: 0x%08lx (size 0x%1x),\n\t virtual addr: 0x%08lx\n", asctime (&when),
						count,
						cdp->memaddr.LowPart,
						cdp->memaddr.HighPart,
						cdp->memsize,
						(unsigned long)cdp->memptr);
				}
			}
		}


		// find I/O base address of the card, assume it is in decoder #2 (=BAR1)
		C(BestTargetDecoderRead(cdp->handle, iodecoder));
		if (cdp->bitver == BEST_32_BIT_CARD)
		{
			C(BestTargetDecoderPropGet(cdp->handle, B_DEC_MODE, &mode));			//changed from MODE to LOCATION
		}
		else
		{
			C(BestTargetDecoderPropGet(cdp->handle, B_DEC_LOCATION, &mode));		//changed from MODE to LOCATION
			C(BestTargetDecoderPropGet(cdp->handle, B_DEC_BASEADDR_HI, &(cdp->ioaddr.HighPart)));
		}

		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_SIZE, &(cdp->iosize)));
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_BASEADDR, &(cdp->ioaddr.LowPart)));
		
		if(cdp->bitver == BEST_32_BIT_CARD)
		{
			if (mode != B_MODE_IO || cdp->ioaddr.LowPart == 0)
			{
				printf("***** Card #%d: BAR1 is not an I/O base address or not assigned by the BIOS\n", count);
				fprintf(stream, "%s ***** Card #%d: BAR1 is not an I/O base address or not assigned by the BIOS\n", asctime (&when), count);
				cdp->ioptr = NULL;
				cdp->ioaddr.LowPart = 0;
				cdp->ioaddr.HighPart = 1;
				cdp->iosize = 0;
				//ret = -1;
			}

			else
			{	cdp->ioaddr.HighPart = 0x0;
				cdp->iosize = 1 << cdp->iosize;
				cdp->ioptr = (b_int8 *)map_physical_address(cdp->ioaddr.LowPart, cdp->ioaddr.HighPart, cdp->iosize, (cdp->devid >> 8) & 0xff, 1);
				if (cdp->ioptr == NULL)
				{
					printf("***** Card #%d: can't map I/O space into virtual memory\n", count);
					fprintf(stream, "%s ***** Card #%d: can't map I/O space into virtual memory.\n", asctime (&when), count);
				}
	
				if (dbg)
				{
					printf("Card #%d: I/O base addr_low: 0x%08lx addr_hi: 0x%08lx (size 0x%lx),\n\t virtual addr: 0x%08lx\n",
						count, 
						cdp->ioaddr.LowPart,
						cdp->ioaddr.HighPart,
						cdp->iosize,
						(unsigned long)cdp->ioptr);
					fprintf(stream, "%s Card #%d: I/O base addr_low: 0x%08lx addr_hi: 0x%08lx (size 0x%1x),\n\t virtual addr: 0x%08lx\n", asctime (&when),
						count, 
						cdp->ioaddr.LowPart,
						cdp->ioaddr.HighPart,
						cdp->iosize,
						(unsigned long)cdp->ioptr);
				}
			}
		}
		else //64 bit card only
		{
			if (mode != B_LOC_IO || cdp->ioaddr.LowPart == 0)
			{
				printf("***** Card #%d: BAR1 is not an I/O base address or not assigned by the BIOS\n", count);
				fprintf(stream, "%s ***** Card #%d: BAR1 is not an I/O base address or not assigned by the BIOS\n", asctime (&when), count);
				cdp->ioptr = NULL;
				cdp->ioaddr.LowPart = 0;
				cdp->ioaddr.HighPart = 0;
				cdp->iosize = 0;
				//ret = -1;
			}

			else
			{
				cdp->iosize = 1 << cdp->iosize;
				cdp->ioptr = (b_int8 *)map_physical_address(cdp->ioaddr.LowPart, cdp->ioaddr.HighPart, cdp->iosize, (cdp->devid >> 8) & 0xff, 1);
				if (cdp->ioptr == NULL)
				{
					printf("***** Card #%d: can't map I/O space into virtual memory\n", count);
					fprintf(stream, "%s ***** Card #%d: can't map I/O space into virtual memory.\n", asctime (&when), count);
				}
	
				if (dbg)
				{
					printf("Card #%d: I/O base addr_low: 0x%08lx addr_hi: 0x%08lx (size 0x%lx),\n\t virtual addr: 0x%08lx\n",
						count,
						cdp->ioaddr.LowPart,
						cdp->ioaddr.HighPart,
						cdp->iosize,
						(unsigned long)cdp->ioptr);
					fprintf(stream, "%s Card #%d: I/O base addr_low: 0x%08lx addr_hi: 0x%08lx (size 0x%1x),\n\t virtual addr: 0x%08lx\n", asctime (&when),
						count,
						cdp->ioaddr.LowPart,
						cdp->ioaddr.HighPart,
						cdp->iosize, 
						(unsigned long)cdp->ioptr);
				}
			}
		}

		//-------------------------------------------------------------------------------------
		// Allocate a block of system memory for each card so that each card
		// has a "playground" for write/read testing
		// Note, that we can reserve a maximum of 4K with a single
		// call, otherwise there is no guarantee, that the memory region is
		// physically contignuous
		//-------------------------------------------------------------------------------------

		if (allocate_physical_memory(page_size, &(cdp->sysmem_ptr), &(cdp->sysmem_addr.LowPart)) != 0)
		{
			printf("***** allocating physical memory failed\n");
			fprintf(stream, "%s ***** allocating physical memory failed.\n", asctime (&when));
			cdp->sysmem_ptr = NULL;
			cdp->sysmem_addr.LowPart = 0;
			cdp->sysmem_addr.HighPart = 0;
			ret = -1;
		}

		else
		{	if (dbg)
			{	printf("Allocated %ldk memory at virtual addr 0x%08lx, physical addr 0x%08lx\n",
					page_size / 1024, 
					(unsigned long)cdp->sysmem_ptr, 
					(unsigned long)cdp->sysmem_addr.LowPart);
				fprintf(stream, "%s Allocated %ldk memory at virtual addr 0x%08lx, physical addr 0x%08lx\n", 
					asctime (&when),
					page_size / 1024, 
					(unsigned long)cdp->sysmem_ptr, 
					(unsigned long)cdp->sysmem_addr.LowPart);
			}
		}
	}

	// initialize PPR, but only ONCE for all cards. There is only one global PPR object
//	if (use_ppr && n_cards > 0)
//	{	C(BestPprInit(card_info[0].handle));
//	}

	for (count = 0, cdp = card_info; count < n_cards; count++, cdp++)
	{	// set up master attribute permutations
		setup_master_attributes(cdp->handle,cdp->bitver);

		// set up target attribute permutations
		setup_target_attributes(cdp->handle,cdp->bitver);

		// set display back to system
		//C(BestDisplayPropSet(cdp->handle, B_DISP_CARD));
		
		// set the analyzer so it triggers on protocol errors
		// C(BestTracePropSet(cdp->handle,B_TRC_PARFANALYZER_MODE,B_E2925_COMPATIBLE));
		
//		C(BestTracePattPropSet(cdp->handle,B_PT_TRIGGER,"berr"));
//		C(BestTracePattPropSet(cdp->handle,B_PT_SQ,"1"));
//
//		C(BestAnalyzerRun(cdp->handle));
	}

	return(ret);
}

//---------------------------------------------------------------------------------------------
// close_all_cards stops all cards, deallocates any mapped memory and disconnects the cards
//---------------------------------------------------------------------------------------------

void close_all_cards()
{	unsigned int i;
	card_info_t *cdp;
	b_errtype err;
	b_handletype handle;

	// stop all cards
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{	handle = cdp->handle;
		C(BestMasterStop(cdp->handle));
	}

	// free PPR resources
//	if (use_ppr && n_cards > 0)
//	{	C(BestPprDelete(cdp->handle));
//	}

	// free all allocated memory and disconnect cards
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{	handle = cdp->handle;
		if (cdp->sysmem_ptr)
		{	free_physical_memory(cdp->sysmem_ptr, page_size);
		}

		if (cdp->memptr)
		{	unmap_physical_address(cdp->memptr);
		}

		if(cdp->bitver == BEST_32_BIT_CARD)
		{
			C(BestDisconnect(cdp->handle));
		}
		C(BestClose(cdp->handle));
	}
}

//---------------------------------------------------------------------------------------------
// TEST 1 - 
// HPBEST reads from system memory 
//---------------------------------------------------------------------------------------------

DWORD __stdcall
mem_read(void *tip)
{	b_errtype err;
	double elapsed;
	card_info_t *cdp;
	char buffer[20];
	LARGE_INTEGER addr;
	b_int32 len;
	b_handletype handle;
	struct tm when;
	time_t now, mem_start_run, mem_end_run;

	time(&now);
	when = *localtime(&now);

	printf("Memory Read Test started %s\n", asctime (&when));
	fprintf(stream, "Memory Read Test started %s\n",asctime (&when));

	fprintf(stream,"\n%s BUS MASTER MEMORY READ TEST\n\n", asctime(&when));
	time (&mem_start_run);

	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}

	handle = cdp->handle;
	printf("Enter base address of memory to read: ");
	gets(buffer);
	addr.LowPart = strtoul(buffer, 0, 0);
	addr.HighPart = 0;
inp2:
	printf("Enter block length of memory read:    ");
	gets(buffer);
	len = strtoul(buffer, 0, 0);
    if(len == 0)
	{
		printf("You must specify a lenght bigger than 0. Please try again.\n");
		goto inp2;
	}

	C(BestMasterGenPropDefaultSet(cdp->handle));
	C(BestMasterGenPropSet(cdp->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));


	setup_master_block(cdp->handle, BPPR_DIR_READ, 
		addr, 0x0000, len, 1, 0, 0,cdp->bitver,WRITE_PPR_REP,0,1);
	C(BestMasterBlockPageRun(cdp->handle, 1));

	printf("mem_read running...\n");
	fprintf(stream, "%s Mem_read_running...\n", asctime (&when));
	((thread_info_t *)tip)->state = RUNNING;

	while (1)
	{	Sleep(500);
		if (check_card(0, cdp))
		{	break;
		}

		if (((thread_info_t *)tip)->term_request)
		{	break;
		}

		time(&mem_end_run);
		elapsed = difftime(mem_end_run,mem_start_run);

		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{	break;
		}
	}

	C(BestMasterStop(cdp->handle));
	cdp->busy = 0;
	((thread_info_t *)tip)->state = TERMINATED;
	printf("mem_read terminating...\n");
	fprintf(stream, "%s Mem_read terminating...\n", asctime (&when));
	time(&mem_end_run);
	elapsed = difftime(mem_end_run,mem_start_run);
	printf("Elapsed time of memory read test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of memory read test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
// TEST 2 - 
// mem_test allocates a block of physical memory, fills it with random data (from the CPU
// side) and instructs the E2926A to read the data (through PCI). Afterwards data is uploaded
// from the E2926A and compared with the original data
//---------------------------------------------------------------------------------------------

DWORD __stdcall
mem_write_read_compare(void *tip)
{	b_handletype handle;
	double elapsed;
	b_errtype err;
	LARGE_INTEGER tmp;
	unsigned long j;
	unsigned long size;
	card_info_t *cdp;
	volatile unsigned char *xbp;
	const unsigned long int_addr1 = 0x200;
	const unsigned long int_addr2 = 0x400;
	const unsigned long int_addr3 = 0x600;
	struct tm when;
	time_t now, rw_compare_start_run, rw_compare_end_run;

	time(&now);
	when = *localtime(&now);

	printf("memory W/R/C started %s\n", asctime (&when));
	fprintf(stream, "Memory W/R/C started %s\n",asctime (&when));

	fprintf(stream,"\n%s MEMORY WRITE/READ/COMPARE TO SYSTEM MEMORY\n\n", asctime(&when));
	time (&rw_compare_start_run);

	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}
	
	handle = cdp->handle;
	size = 0x200;

	// initialize the internal memory of the cards through the mapped address space
	for (j = 0, xbp = ((char *)cdp->memptr) + int_addr1; j < size; j++, xbp++)
	{	*xbp = 0x5a;
	}

	for (j = 0, xbp = ((char *)cdp->memptr) + int_addr2; j < size; j++, xbp++)
	{	*xbp = rand();
	}

	tmp = cdp->sysmem_addr;                                         // write to the allocated system memory

	printf("setup card #%d to r/w to physical addr 0x%08lx\n", cdp - card_info, tmp);
	fprintf(stream, "%s Setup card #%d to r/w to physical addr 0x%08lx\n", asctime (&when), cdp - card_info, tmp);

	
	C(BestMasterGenPropDefaultSet(cdp->handle));
	C(BestMasterGenPropSet(cdp->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));

	// set up block transfer to write/read/compare data
	setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, int_addr1, size,1, 0, 0,cdp->bitver,WRITE_PPR_REP,0,0);
	setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, int_addr3, size,BPPR_BLK_APPEND, 1, int_addr1,cdp->bitver,WRITE_PPR_REP,1,0);

	// set up block transfer to write/read/compare data from a different internal address
	setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, int_addr2, size,BPPR_BLK_APPEND, 0, 0,cdp->bitver,WRITE_PPR_REP,2,0);
	setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, int_addr3, size,BPPR_BLK_APPEND, 1, int_addr2,cdp->bitver,WRITE_PPR_REP,3,1);

	C(BestMasterBlockPageRun(cdp->handle, 1));

	((thread_info_t *)tip)->state = RUNNING;

	while (1)
	{	Sleep(500);
		if (check_card(0, cdp))
		{	break;
		}

		if (((thread_info_t *)tip)->term_request)
		{	break;
		}

		time(&rw_compare_end_run);
		elapsed = difftime(rw_compare_end_run,rw_compare_start_run);
		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{	break;
		}
	}

	C(BestMasterStop(cdp->handle));	
	((thread_info_t *)tip)->state = TERMINATED;
	printf("memory_write_read_compare terminating...\n");
	fprintf(stream, "%s memory_write_read_compare terminating...\n", asctime (&when));
	time (&rw_compare_end_run);
	elapsed = difftime(rw_compare_end_run,rw_compare_start_run);
	printf("Elapsed time of memory_read_write_compare test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of memory_read_write_copare test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}


//---------------------------------------------------------------------------------------------
// TEST 3 - 
// CPU accesses memory space on HPBEST card
//---------------------------------------------------------------------------------------------

DWORD __stdcall
cpu_mem_test(void *tip)
{	card_info_t *cdp;
	b_errtype err;
	double elapsed;
	unsigned long *cmp_buf;
	b_int32 size, i;
	int cont = 1;
	b_handletype handle;
	volatile unsigned char *xbp;
	volatile unsigned short *xsp;
	volatile unsigned long *xlp;
	struct tm when;
	time_t now,cpu_start_run,cpu_end_run;

	time(&now);
	when = *localtime(&now);

	printf("CPU Memory Test started %s\n", asctime (&when));
	fprintf(stream, "CPU Memory Test started %s\n",asctime (&when));

	fprintf(stream,"\n%s CPU ACCESS to HPBEST MEMORY SPACE\n\n", asctime(&when));
	time (&cpu_start_run);

	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}

	handle = cdp->handle;
	size = cdp->memsize;
	if (size > page_size)
	{	size = page_size;
	}

	size = 16;

	printf("block size = 0x%lx\n", size);
	fprintf(stream, "block size = 0x%lx\n", size);
	printf("virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->memptr, cdp->memaddr.LowPart);
	fprintf(stream, "virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->memptr, cdp->memaddr.LowPart);

	cmp_buf = malloc(4*size);
	if (cdp->memptr)
	{	((thread_info_t *)tip)->state = RUNNING;
		while (cont)
		{	for (i = 0, xbp = cdp->memptr; i < size; i++, xbp++)
			{	cmp_buf[i] = *xbp = rand();
			}
			
			for (i = 0, xbp = cdp->memptr; i < size; i++, xbp++)
			{	if (*xbp != cmp_buf[i])
				{	printf("***** cpu_mem_test: failed with BYTES at offset 0x%08lx\n", i);
					fprintf(stream,"%s ***** cpu_mem_test: failed with BYTES at offset 0x%08lx\n", asctime (&when), i);
					cont = 0;
					break;
				}
				else
				{	//TODO Remove for Will b/c he don't wanna see this on-screen (5-15-98)
					printf("cpu_mem_test: Wrote: 0x%02lX to 0x%08lx and Read: 0x%02lX - Pass\n", cmp_buf[i], cdp->memptr, *xbp);
					fprintf(stream,"%s cpu_mem_test: Wrote: 0x%02lX to 0x%08lx and Read: 0x%02lX - Pass\n",asctime(&when), cmp_buf[i], cdp->memptr, *xbp);
				}
			}

			for (i = 0, xsp = cdp->memptr; i < size / 2; i++, xsp++)
			{	cmp_buf[i] = *xsp = rand();
			}

			for (i = 0, xsp = cdp->memptr; i < size / 2; i++, xsp++)
			{	if (*xsp != cmp_buf[i])
				{	printf("***** cpu_mem_test: failed with WORDS at offset 0x%08lx\n", i);
					fprintf(stream,"%s ***** cpu_mem_test: failed with WORDS at offset 0x%08lx\n", asctime (&when), i);
					cont = 0;
					break;
				}
				else
				{	//TODO Remove for Will b/c he don't wanna see this on-screen (5-15-98)
					printf("cpu_mem_test: Wrote: 0x%04lX to 0x%08lx and Read: 0x%04lX - Pass\n", cmp_buf[i], cdp->memptr, *xsp);
					fprintf(stream,"%s cpu_mem_test: Wrote: 0x%04lX to 0x%08lx and Read: 0x%04lX - Pass\n",asctime(&when), cmp_buf[i], cdp->memptr, *xsp);
				}
			}

			for (i = 0, xlp = cdp->memptr; i < size / 4; i++, xlp++)
			{	cmp_buf[i] = *xlp = (rand() | rand() << 16);
			}

			for (i = 0, xlp = cdp->memptr; i < size / 4; i++, xlp++)
			{	if (*xlp != cmp_buf[i])
				{	printf("***** cpu_mem_test: failed with DWORDS at offset 0x%08lx\n", i);
					fprintf(stream,"%s ***** cpu_mem_test: failed with DWORDS at offset 0x%08lx\n", asctime (&when), i);
					cont = 0;
					break;
				}
				else
				{	//TODO Remove for Will b/c he don't wanna see this on-screen (5-15-98)
					printf("cpu_mem_test: Wrote: 0x%08lX to 0x%08lx and Read: 0x%08lX - Pass\n", cmp_buf[i], cdp->memptr, *xlp);
					fprintf(stream,"%s cpu_mem_test: Wrote: 0x%08lX to 0x%08lx and Read: 0x%08lX - Pass\n",asctime(&when), cmp_buf[i], cdp->memptr, *xlp);
				}
			}

			if (check_card(0, cdp))
			{	break;
			}

			if (((thread_info_t *)tip)->term_request)
			{	break;
			}

			Sleep(500);

			time(&cpu_end_run);
			elapsed = difftime(cpu_end_run, cpu_start_run);

			if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
			{	break;
			}
		}
	}

	else
	{	printf("***** can't access on-board memory\n");
		fprintf(stream, "%s ***** Can't access on-board memory.\n", asctime (&when));
	}

	free(cmp_buf);
	cdp->busy = 0;
	C(BestMasterStop(cdp->handle));	
	((thread_info_t *)tip)->state = TERMINATED;
	printf("cpu_mem_test terminating... (maxtime=%g)\n", ((thread_info_t *)tip)->maxtime);
	fprintf(stream, "%s cpu_mem_test terminating...\n", asctime (&when));
	time (&cpu_end_run);
	elapsed = difftime(cpu_end_run,cpu_start_run);
	printf("Elapsed time of cpu_mem_test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of cpu_mem_test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
// TEST 4 - 
// CPU accesses I/O space on HPBEST card
//---------------------------------------------------------------------------------------------

DWORD __stdcall
cpu_io_test(void *tip)
{	b_errtype err;
	card_info_t *cdp;
    double elapsed;
	unsigned long *cmp_buf;
	b_int32 size;
	b_int32 i;
	int cont = 1;
	b_handletype handle;
	unsigned char *xbp;
	unsigned long xl;
    struct tm when;
    time_t now, cpu_IO_end_run, cpu_IO_start_run;
	unsigned long countB = 0;
	unsigned long countW = 0;
	unsigned long countD = 0;

	time(&now);
    when = *localtime(&now);

	printf("CPU IO Test started %s\n", asctime (&when));
	fprintf(stream, "CPU IO Test started %s\n",asctime (&when));
    
	fprintf(stream,"\n%s CPU ACCESS to HPBEST I/O SPACE\n\n", asctime(&when));
    time (&cpu_IO_start_run);

	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}

	handle = cdp->handle;
	size = cdp->iosize;
	if (size > 16)
	{	size = 16;
	}

    printf("block size = 0x%lx\n", size);
    fprintf(stream, "block size = 0x%lx\n", size);
    printf("virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->ioptr, cdp->ioaddr.LowPart);
    fprintf(stream, "virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->ioptr, cdp->ioaddr.LowPart);

	cmp_buf = malloc(size * sizeof(unsigned long));
	if (cdp->ioptr)
	{	((thread_info_t *)tip)->state = RUNNING;

		while (cont)
		{
		//	Sleep(500);

			for (i = 0,  xbp = cdp->ioptr; i < size; i++, xbp += 1)
			{	xl = cmp_buf[i] = rand();
				io_access(BEST_NTMEM_IO_WRITE_BYTE, (unsigned long)xbp, &xl);
			}

			for (i = 0,  xbp = cdp->ioptr; i < size; i++, xbp += 1)
			{	io_access(BEST_NTMEM_IO_READ_BYTE, (unsigned long)xbp, &xl);
				++countB;
				if (xl != (cmp_buf[i] & 0x000000ff))
				{	printf("***** cpu_io_test: failed with BYTES at offset 0x%08lx Wrote:%d, Read: %d\n", i,cmp_buf[i],xl);
				    fprintf(stream,"***** cpu_io_test: failed wait BYTES at offset 0x%08lx\n", i);
					cont = 0;
					break;
				}
				
				else
				{	//TODO Remove for Will b/c he don't wanna see this on-screen
					printf("cpu_io_test Wrote(Byte): 0x%02lX to 0x%08lx and Read: 0x%02lX - Pass\n", (cmp_buf[i] & 0x000000ff), cdp->ioptr, xl);
					fprintf(stream,"%s cpu_io_test Wrote(Byte): 0x%02lX to 0x%08lx and Read: 0x%02lX - Pass\n",asctime(&when), (cmp_buf[i] & 0x000000ff), cdp->ioptr, xl);
				}
				Sleep(100);
			}

			for (i = 0,  xbp = cdp->ioptr; i < size / 2; i++, xbp += 2)
			{	xl = cmp_buf[i] = rand();
				io_access(BEST_NTMEM_IO_WRITE_WORD, (unsigned long)xbp, &xl);
			}

			for (i = 0,  xbp = cdp->ioptr; i < size / 2; i++, xbp += 2)
			{	io_access(BEST_NTMEM_IO_READ_WORD, (unsigned long)xbp, &xl);
				++countW;
				if (xl != (cmp_buf[i] & 0x0000ffff))
				{	printf("***** cpu_io_test: failed with WORDS at offset 0x%08lx\n", i);
				    fprintf(stream,"***** cpu_io_test: failed wait WORDS at offset 0x%08lx\n", i);
					cont = 0;
					break;
				}
				else
				{	//TODO Remove for Will b/c he don't wanna see this on-screen
					printf("cpu_io_test Wrote(WORD): 0x%04lX to 0x%08lx and Read: 0x%04lX - Pass\n", (cmp_buf[i] & 0x0000ffff), cdp->ioptr, xl);
					fprintf(stream,"%s cpu_io_test Wrote: 0x%04lX to 0x%08lx and Read: 0x%04lX - Pass\n",asctime(&when), (cmp_buf[i] & 0x0000ffff), cdp->ioptr, xl);
				}
				Sleep(100);
			}

			for (i = 0,  xbp = cdp->ioptr; i < size / 4; i++, xbp += 4)
			{	xl = cmp_buf[i] = rand();
				io_access(BEST_NTMEM_IO_WRITE_DWORD, (unsigned long)xbp, &xl);
			}

			for (i = 0,  xbp = cdp->ioptr; i < size / 4; i++, xbp += 4)
			{	io_access(BEST_NTMEM_IO_READ_DWORD, (unsigned long)xbp, &xl);
				++countD;
				if (xl != cmp_buf[i])
				{	printf("***** cpu_io_test: failed with DWORDS at offset 0x%08lx\n", i);
				    fprintf(stream,"***** cpu_io_test: failed wait DWORDS at offset 0x%08lx\n", i);
					cont = 0;
					break;
				}
				else
				{	//TODO Remove for Will b/c he don't wanna see this on-screen
					printf("cpu_io_test Wrote(DWORD): 0x%08lX to 0x%08lx and Read: 0x%08lX - Pass\n", cmp_buf[i], cdp->ioptr, xl);
					fprintf(stream,"%s cpu_io_test Wrote(DWORD): 0x%08lX to 0x%08lx and Read: 0x%08lX - Pass\n",asctime(&when), cmp_buf[i], cdp->ioptr, xl);
				}		
				Sleep(100);
			}

			if (check_card(0, cdp))
			{	break;
			}

			if (((thread_info_t *)tip)->term_request)
			{	break;
			}

			time(&cpu_IO_end_run);
		    elapsed = difftime(cpu_IO_end_run, cpu_IO_start_run);
	
			if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		    {	break;
			}
		}
	}

	else
    {   printf("***** can't access on-board I/O space\n");
	    fprintf(stream, "%s ***** Can't access on-board I/O space.\n", asctime (&when));
    }

	printf("%d Bytes \n%d Words \n%d DWords \n%d Total\n", countB, countW, countD, (countB+countW+countD));
    fprintf(stream,"%s %d Bytes \n%d Words \n%d DWords \n%d Total\n", asctime(&when), countB, countW, countD, (countB+countW+countD));
	free(cmp_buf);
    ((thread_info_t *)tip)->state = TERMINATED;
    C(BestMasterStop(cdp->handle));	
	printf("cpu_io_test terminating...\n");
    fprintf(stream, "cpu_io_test terminating...\n");
    time (&cpu_IO_end_run);
    elapsed = difftime(cpu_IO_end_run,cpu_IO_start_run);
    printf("Elapsed time of cpu_io_test: %.2f minutes.\n", elapsed / 60.0);
    fprintf(stream,"Elapsed time of cpu_io_test: %.2f minutes.\n", elapsed / 60.0);
    return(0);
}

//---------------------------------------------------------------------------------------------
// TEST 5 -
// peer2peer locates two E2926A's in the system. One will be used as a master, the other
// one as a target in order to do peer-to-peer traffic on PCI. The two E2926A's don't need
// to be on the same PCI bus. If they are on different buses, the traffic will be sent
// through a PCI-to-PCI bridge.
// peer2peer fills the internal memory of the first card with random data, sets up the
// first card to write to the memory space of the second card, reads the data back
// and compares. Then the same thing is performed with exchanged roles.
//---------------------------------------------------------------------------------------------

DWORD __stdcall
peer2peer(void *tip)
{	b_errtype err,err2;
	double elapsed;
	unsigned long found = 0;
	int fail = 0;
	b_int32 size;
	card_info_t *cdp1, *cdp2;
	struct tm when;
	time_t now, peer2peer_start_run, peer2peer_end_run;
	b_int32 status;
	int contflag = 1;

	time(&now);
	when = *localtime(&now);

	printf("Peer to Peer Test started %s\n", asctime (&when));
	fprintf(stream, "Peer to Peer Test started %s\n",asctime (&when));

	fprintf(stream,"\n%s PEER to PEER READ/WRITE TEST\n\n", asctime(&when));
	time (&peer2peer_start_run);

	if ((cdp1 = select_single_card(tip, "Select first card:  ")) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}

	if ((cdp2 = select_single_card(tip, "Select second card: ")) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}

	// set size to the minimum of the two memory sizes
	size = cdp1->memsize;
	if (cdp2->memsize < size)
	{	size = cdp2->memsize;
	}

	if (size > 0x4000)
	{	size = 0x4000;
	}

	printf("\nUsing cards #%d and #%d for peer-to-peer test\n",
		cdp1 - card_info, cdp2 - card_info);
	printf("block size: 0x%lx\n", size);
	printf("physical address: card %d:0x%08lx, card %d:0x%08lx\n",
		cdp1 - card_info, cdp1->memaddr.LowPart, cdp2 - card_info, cdp2->memaddr.LowPart);

    fprintf(stream, "\n%s Using cards #%d and #%d for peer-to-peer test.\n",
	    asctime (&when), cdp1 - card_info, cdp2 - card_info);
    fprintf(stream, "block size: 0x%lx\n", size);
    fprintf(stream, "physical address: card %d:0x%08lx, card %d:0x%08lx\n",
	    cdp1 - card_info, cdp1->memaddr.LowPart, cdp2 - card_info, cdp2->memaddr.LowPart);

	// setup card1 to write/read/compare card2's memory
	setup_master_block(cdp1->handle, BPPR_DIR_WRITE, cdp2->memaddr, 0x10000, size, 1, 0, 0,cdp1->bitver,WRITE_PPR_REP,0,0);
	setup_master_block(cdp1->handle, BPPR_DIR_READ,  cdp2->memaddr, 0x14000, size, BPPR_BLK_APPEND, 1, 0x10000,cdp1->bitver,WRITE_PPR_REP,1,0);
	setup_master_block(cdp1->handle, BPPR_DIR_WRITE, cdp2->memaddr, 0x10008, size-8, BPPR_BLK_APPEND, 0, 0,cdp1->bitver,WRITE_PPR_REP,2,0);
	setup_master_block(cdp1->handle, BPPR_DIR_READ,  cdp2->memaddr, 0x14000, size-8, BPPR_BLK_APPEND, 1, 0x10008,cdp1->bitver,WRITE_PPR_REP,3,1);
	C(BestMasterGenPropSet(cdp1->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));

	C(BestObsStatusClear(cdp1->handle));
	C(BestObsStatusClear(cdp2->handle));
	C(BestStatusRegClear(cdp1->handle,0xffffffff));
	C(BestStatusRegClear(cdp2->handle,0xffffffff));


	err2 = BestMasterBlockPageRun(cdp1->handle, 1);
	if(err2 == B_E_FUNC)
	{	status = 0;
		C(BestStatusRegGet(cdp1->handle, &status));	
		if(status & 0x80)
		{
			printf("Master Abort detected! The other card does not respond \nor the call is not propagated through the bridge(s).\n");
			fprintf(stream,"Master Abort detected! The other card does not respond \nor the call is not propagated through the bridge(s).\n");
		    contflag = 0;
		}
		else
		{
			printf("Functional Onboard Error occured. (Line: %d)\n",__LINE__);
			fprintf(stream,"Functional Onboard Error occured. (Line: %s)\n",__LINE__);
		}
	}
	else
	{
		if(err2 != B_E_OK)
		    printf("%s (line %d)\n", BestErrorStringGet(err2), __LINE__);
	}

//---------------------------------------------------------------------------------------------
//      // can't run the other card in parallel for now, because of master/target limitations
//      setup_master_block(cdp2->handle, BPPR_DIR_WRITE, cdp1->memaddr, 0x10000, size, 1, 0, 0);
//      setup_master_block(cdp2->handle, BPPR_DIR_READ,  cdp1->memaddr, 0x14000, size, BPPR_BLK_APPEND, 1, 0x10000);
//      setup_master_block(cdp2->handle, BPPR_DIR_WRITE, cdp1->memaddr, 0x10004, size, BPPR_BLK_APPEND, 0, 0);
//      setup_master_block(cdp2->handle, BPPR_DIR_READ,  cdp1->memaddr, 0x14000, size, BPPR_BLK_APPEND, 1, 0x10004);
//      C(BestMasterGenPropSet(cdp2->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));
//      C(BestMasterBlockPageRun(cdp2->handle, 1));
//----------------------------------------------------------------------------------------------
	
	((thread_info_t *)tip)->state = RUNNING;
	while (contflag)
	{	Sleep(500);
		if (check_card(0, cdp1))
		{	break;
		}

		if (check_card(0, cdp2))
		{	break;
		}

		if (((thread_info_t *)tip)->term_request)
		{	break;
		}

		time(&peer2peer_end_run);
		elapsed = difftime(peer2peer_end_run, peer2peer_start_run);
	
		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{	break;
		}
	}

	((thread_info_t *)tip)->state = TERMINATED;
	C(BestMasterStop(cdp1->handle));
	C(BestMasterStop(cdp2->handle));
	printf("peer2peer terminating...\n");
	fprintf(stream, "peer2peer teminating...\n");
	time (&peer2peer_end_run);
	elapsed = difftime(peer2peer_end_run,peer2peer_start_run);
	printf("Elapsed time of peer2peer test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of peer2peer test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
// TEST 6 -
// config_scan scans the PCI bus in which the E2926A is plugged in. It reads the vendor ID/
// device ID register and the class code and prints it out
//---------------------------------------------------------------------------------------------

DWORD __stdcall
config_scan(void *tip)
{	b_handletype handle;
	double elapsed;
	b_errtype err;
	b_int32 val, class_code;
	unsigned long i, j, found = 0;
	char *s;
	card_info_t *cdp;
	struct tm when;
	time_t now, config_scan_start_run, config_scan_end_run;

	time(&now);
	when = *localtime(&now);

	printf("Config Scan started %s\n", asctime (&when));
	fprintf(stream, "Config Scan started %s\n",asctime (&when));

	fprintf(stream,"\n%s PCI CONFIG SCAN TEST\n\n", asctime(&when));
	time (&config_scan_start_run);

	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}

	handle = cdp->handle;
	for (i = 0x800; i; i <<= 1)                      // loop through all devices on this PCI bus
	{	for (j = 0; j < 0x800; j += 0x100) // loop through all functions within a device
		{	// read vendor ID/device ID register (config space offset 0)
			err = BestHostPCIRegGet(handle, B_ASP_CONFIG, i+j+0, &val, 4);
			if (err == B_E_MASTER_ABORT || val == 0xffffffff)
			{	// master abort or 0xffffffff is normal for non-existing devices
			}
	
			else if (err != B_E_OK)
			{	printf("BestHostPCIRegGet failed (%s)\n", BestErrorStringGet(err));
				fprintf(stream, "%s BestHostPCIRegGet failed (%s)\n", asctime (&when), BestErrorStringGet(err));
			}

			else
			{	// read class code register (config space offset 8)
				C(BestHostPCIRegGet(handle, B_ASP_CONFIG, i+j+8, &class_code, 4));
				// class code is in bits 8...31
				class_code >>= 8;
				switch(class_code & 0xff0000)
				{	case 0x000000: s = "Device was built before class codes were defined"; break;
		
					case 0x010000: s = "Mass storage controller"; break;

					case 0x020000: s = "Network controller"; break;

					case 0x030000: s = "Display controller"; break;

					case 0x040000: s = "Multimedia controller"; break;

					case 0x050000: s = "Memory controller"; break;

					case 0x060000: s = "Bridge device"; break;

					case 0x070000: s = "Simple communication controller"; break;

					case 0x080000: s = "Base system peripheral"; break;

					case 0x090000: s = "Input device"; break;

					case 0x0a0000: s = "Docking station"; break;

					case 0x0b0000: s = "Processor"; break;

					case 0x0c0000: s = "Serial bus controller"; break;

					case 0xff0000: s = "Device does not fit into any defined class"; break;

					default:       s = "Reserved class code"; break;
				}

				if (found == 0)
				{	printf("Cfg.Addr VendorID DeviceID Class\n");
					printf("--------------------------------\n");
					printf("Current time is: %s\n", asctime (&when));
					printf("-----------------------------------------\n");
					fprintf(stream, "Cfg.Addr VendorID DeviceID Class\n");
					fprintf(stream, "--------------------------------\n");
					fprintf(stream, "Current time is: %s\n", asctime (&when));
					fprintf(stream, "-----------------------------------------\n");
				}
	
				printf("%08lx   %04lx     %04lx   %s\n", i+j, val >> 16, val & 0xffff, s);
				fprintf(stream,"%08lx   %04lx     %04lx   %s\n", i+j, val >> 16, val & 0xffff, s);
				
				// Read cache line size (config space offset 0x0C)
				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x0C, &val, 1);
				
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
				{	// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
				
				else
				{	printf("\nThe cache line size is: %02lx.\n\n", val);
					fprintf(stream, "\nThe cache line size is: %02lx.\n\n", val);
				}
				
				// Read Base Addr register 0 (config space offset 0x10)
				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x10, &val, 4);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
			    {	// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
            
				else
				{	// Bit 0:  Address space
					printf ("According to the Base Address register 0:\n"); 
					fprintf (stream, "According to the Base Address register 0:\n");
					if (val & 0x0001 == 0x1)
					{	printf ("  Address space enabled.\n");
						fprintf (stream, "  Address space enabled.\n");
					}
			   
					else 
						printf ("  Address Space disabled.\n");
						fprintf (stream, "  Address Space disabled.\n");

					// Bit 1:  Memory space location
					if (val & 0x0002 == 0x1)
					{	printf ("  Memory space location enabled.\n");
						fprintf (stream, "  Memory space location enabled.\n");
					}
					
					else
			   			printf ("  Memory space location disabled.\n");
						fprintf (stream, "  Memory space location disabled.\n");

					// Bit 2:  Memory space location
					if (val & 0x0004 == 0x1) 
					{	printf ("  Memory space location enabled.\n");
						fprintf (stream, "  Memory space location enabled.\n");
					}
					
					else 
			   		printf ("  Memory space location disabled.\n");
					fprintf (stream, "  Memory space location disabled.\n");
  				}

				// Read Base Addr register 1 (config space ofset 0x14)

				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x14, &val, 4);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
				{	// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
			
				else
				{	// Bit 0:  Address space
					printf ("According to the Base Address register 1:\n");
					fprintf (stream, "According to the Base Address register 1:\n");
					if (val & 0x0001 == 0x1) 
					{	printf ("  Address Space enabled.\n");
						fprintf (stream, "  Address Space enabled.\n");
					}
					else 
						printf ("  Address Space disabled.\n");
						fprintf (stream, "  Address Space disabled.\n");
				}
  	
				// Read Subsystem ID (config space offset 0x2E)

				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x2E, &val, 2);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFF)
				{	// Master abort of 0xFFFFFFFF is normal for non-existing devices.
				}
             
				else
					printf ("\nThe subsystem ID is: %04lx.\n\n", val);
					fprintf (stream, "\nThe subsystem ID is %04lx.\n\n", val);
           
				// Read Command Register (config space ofset 0x04)

				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x04, &val, 2);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
				{	// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
	
				else
				{	// Bit 0: I/O Space Control, RW
					if (val & 0x0001 == 0x1) 
					{	printf ("  IO Space Control is enabled.\n");
						fprintf (stream, "  IO Space Control is enabled.\n");
					}
					else
						printf ("  IO Space Control is disabled.\n");	   
						fprintf (stream, "  IO Space Control is disabled.\n");
			   			   
					// Bit 1: Memory space, RW
					if (val & 0x0002 == 0x1) 
					{	printf ("  Memory space is enabled.\n");
						fprintf (stream, "  Memory space is enabled.\n");
					}
					else
						printf ("  Memory space is disabled.\n");
						fprintf (stream, "  Memory space is disabed.\n");
			   			   
					// Bit 2: Bus Master Control, RW
					if (val & 0x0004 == 0x1) 
					{	printf ("  Bus Master Control is enabled.\n");
						fprintf (stream, "  Bus Master Control is enabled.\n");
					}
					else
						printf ("  Bus Master Control is disabled.\n");
						fprintf (stream, "  Bus Master Control is disabled.\n");
			   
					// Bit 4: Memory Write and Invalidate Enable, RW
					if (val & 0x0008 == 0x1) 
					{	printf ("  Memory Write and Invalidate is enabled.\n"); 
						fprintf (stream, "  Memory Write and Invalidate is enabled.\n");
					}
					else
						printf ("  Memory Write and Invalidate is disabled.\n");
						fprintf (stream, "  Memory Write and Invalidate is disabled.\n");
			   
					// Bit9: Master fast back to back is enabled, RO
					if (val & 0x0100 == 0x1) 
					{	printf ("  Master fast back to back is enabled.\n");
						fprintf (stream, "  Master fast back to back is enabled.\n");
					}
					else
						printf ("Master fast back to back is disabled.\n");
						fprintf (stream, "  Master fast back to back is disabled.\n");
				}

				found++;
				// if we are looking at function #0 of a given device, check whether this
				// is a multifunction device by reading the header type at offset 0x0e
				// (Note the 5th parameter in BestHostPCIRegGet: reading a single byte)
				// If it is a single function device, skip the other functions
				if (j == 0)
				{	C(BestHostPCIRegGet(handle, B_ASP_CONFIG, i+0x0e, &val, 1));
					if ((val & 0x80) == 0)
						break;
				}
			}
		}
	}

	if (found == 0)
	{	printf("No PCI devices found\n");
		fprintf(stream, "No PCI devices found\n");
	}
	cdp->busy = 0;
	((thread_info_t *)tip)->state = TERMINATED;

	time (&config_scan_end_run);
	elapsed = difftime(config_scan_end_run,config_scan_start_run);
	printf("Elapsed time of config_scan test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of config scan test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
// TEST 7 - 
// stress_test locates all E2926A's in the system, allocates a block of physical memory for each
// card, and instructs the E2926A to write/read/compare.
//---------------------------------------------------------------------------------------------

DWORD __stdcall
stress_test(void *tip)
{	b_errtype err;
	double elapsed;
	card_info_t *cdp;
	int cont = 1;
	unsigned long i, j;
	const unsigned long size = 256;         // size of blocks that we will transfer
	const unsigned long int_addr1 = 0x200;
	const unsigned long int_addr2 = 0x400;
	const unsigned long int_addr3 = 0x600;
	const unsigned long int_addr4 = 0x800;
	LARGE_INTEGER tmp;
	volatile unsigned char *xbp;
//  unsigned char cmp_buf[256];
	struct tm when;
	time_t now, stress_start_run, stress_end_run;

	time(&now);
	when = *localtime(&now);

	printf("Stress Test started %s\n", asctime (&when));
	fprintf(stream, "Stress Test started %s\n",asctime (&when));

	fprintf(stream,"\n%s STRESS TEST\n", asctime(&when));
	time (&stress_start_run);

	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{	if (cdp->busy)
		{	continue;
		}

		if (cdp->memptr == NULL)
		{	continue;
		}

		cdp->busy = 1;
		cdp->tip = tip;

		// initialize the internal memory of the cards through the mapped address space
		for (j = 0, xbp = ((char *)cdp->memptr) + int_addr1; j < size; j++, xbp++)
		{	*xbp = 0x5a;
		}

		for (j = 0, xbp = ((char *)cdp->memptr) + int_addr2; j < size; j++, xbp++)
		{	*xbp = rand();
		}

	//	tmp = card_info[(i+1)%n_cards].memaddr.LowPart;			// write to the next card's built-in memory
		tmp = cdp->sysmem_addr;							// write to the allocated system memory

		printf("setup card #%ld to r/w addr 0x%08lx\n", i, tmp);
		fprintf(stream, "%s setup card #%ld to r/w addr 0x%08lx\n", asctime(&when), i, tmp);

		// set up block transfer to read data
		setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, int_addr1, size,
				      1, 0, 0,cdp->bitver,WRITE_PPR_REP,0,0);
		setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, int_addr3, size,
			BPPR_BLK_APPEND, 1, int_addr1,cdp->bitver,WRITE_PPR_REP,1,0);
		setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, int_addr2, size,
			BPPR_BLK_APPEND, 0, 0,cdp->bitver,WRITE_PPR_REP,2,0);
		setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, int_addr3, size,
			BPPR_BLK_APPEND, 1, int_addr2,cdp->bitver,WRITE_PPR_REP,3,1);
	}

	printf("\n");
	fprintf(stream, "\n");

	// run all cards
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{	if (cdp->tip != tip)
		{	continue;
		}

		C(BestMasterGenPropSet(cdp->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));
		if (cdp->sysmem_addr.LowPart)
		{	printf("starting card #%ld\n", i);
			fprintf(stream, "%s Starting card #%ld\n", asctime (&when), i);
			C(BestMasterBlockPageRun(cdp->handle, 1));
		}
	}

	printf("\n");
	fprintf(stream, "\n");

	((thread_info_t *)tip)->state = RUNNING;

	cont = 1;
	while (cont)
	{	Sleep(500);

	//------------------------------------------------------------------------------------------
	//	/* can't write to card in parallel, because of master/target limitations */
	//	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	//	{	for (j = 0, xbp = (char *)cdp->memptr + int_addr4; j < size; j++, xbp++)
	//			cmp_buf[j] = *xbp = rand();
	//		for (j = 0, xbp = (char *)cdp->memptr + int_addr4; j < size; j++, xbp++)
	//		{	if (*xbp != cmp_buf[j])
	//			{	printf("***** stress_test: card #%d failed with BYTES at offset 0x%08lx\n", i, j);
	//				fprintf(stream, "***** stress_test: card #%d failed with BYTES at offset 0x%08lx\n", i, j);
	//				cont = 0;
	//				break;
	//			}
	//		}
	//	}
	//---------------------------------------------------------------------------------------------

		for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
		{	if (cdp->tip != tip)
			{	continue;
			}

			if (check_card(0, cdp))
			{	cont = 0;
			}
		}

		if (((thread_info_t *)tip)->term_request)
		{	cont = 0;
		}

		time (&stress_end_run);
		elapsed = difftime(stress_end_run,stress_start_run);

		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{	break;
		}
	}

	BestMasterStop(cdp->handle);
	((thread_info_t *)tip)->state = TERMINATED;
	printf("stress_test terminating...\n");
	fprintf(stream, "%s Stress_test terminating...\n", asctime (&when));
	time (&stress_end_run);
	elapsed = difftime(stress_end_run,stress_start_run);
	printf("Elapsed time of stress test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of stress test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
//	Spawning function to use multiple threads in one program
//
//---------------------------------------------------------------------------------------------
int spawn(fct_tab_t *ftp)
{	thread_info_t *tip;
	struct tm when;
	time_t now;
	char buf[20];

	time(&now);
	when = *localtime(&now);

	if (n_threads >= MAX_THREADS)
	{	printf("can't open more threads\n");
		fprintf(stream,"%s Can't open more threads.\n", asctime (&when));
		return(-1);
	}

	else
	{	tip = &thread_info[n_threads];
		printf("Number of minutes to run this test? ");
		gets(buf);
		if (*buf)
		{	tip->maxtime = atof(buf);
		}

		else
		{	tip->maxtime = default_maxtime;
		}

		printf("Test is scheduled to run for %g minutes\n", tip->maxtime);
		fprintf(stream,"Test is scheduled to run for %g minutes\n", tip->maxtime);
		tip->state = STARTING;
		tip->term_request = 0;
		tip->ftp = ftp;
		tip->hThread = CreateThread(NULL, 4096, ftp->fct, tip, 0, &(tip->idThread));
		while (tip->state == STARTING)
		{	Sleep(100);
		}
		// just in case the thread terminated immediately, allow some time for messages

		Sleep(1000);
		n_threads++;
		return (0);
	}
}

//---------------------------------------------------------------------------------------------
// TEST 8 - 
// HPBEST reads from system memory and scans through all
// memory and is even able to read above 4 Gig. As I said,
// HPBEST is the best invention since sliced bread.
//---------------------------------------------------------------------------------------------

DWORD __stdcall
mem_scan(void *tip)
{	b_errtype err,err2;
	double elapsed;
	card_info_t *cdp;
	char buffer[20];
	LARGE_INTEGER addr;
	LARGE_INTEGER endaddr,zw;
	b_int32 len,status;
	b_handletype handle;
	struct tm when;
	time_t now, mem_start_run, mem_end_run;
	char * obserr = NULL;
	int errflag = 0;
	int doreadwrite = 0;
	int mycnt;

	time(&now);
	when = *localtime(&now);

	printf("Memory Scan Test started %s\n", asctime (&when));
	fprintf(stream, "Memory Scan Test started %s\n",asctime (&when));

	fprintf(stream,"\n%s BUS MASTER MEMORY SCAN TEST\n\n", asctime(&when));
	time (&mem_start_run);

	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{	((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}

	handle = cdp->handle;

inp:
	printf("Enter base address (low part) of memory where to read: \n");
	gets(buffer);
	addr.LowPart = strtoul(buffer, 0, 0);
    if(cdp->bitver == BEST_64_BIT_CARD)
	{
		printf("Enter base address (high part) of memory where to read (default 0): \n");
		gets(buffer);
		addr.HighPart = strtoul(buffer, 0, 0);
    }
    else
	{
		addr.HighPart = 0;
    }

	printf("Enter end address (low part) of memory where to read: \n");
	gets(buffer);
	endaddr.LowPart = strtoul(buffer, 0, 0);
    if(cdp->bitver == BEST_64_BIT_CARD)
	{
		printf("Enter end address (high part) of memory where to read (default 0): \n");
		gets(buffer);
		endaddr.HighPart = strtoul(buffer, 0, 0);
    }
    else
	{
		endaddr.HighPart = 0;
    }

	if((endaddr.HighPart == 0) ^ (addr.HighPart == 0))
	{
		printf("You cannot read accross the first 4 Gig boundary. Please try again.\n");
	    goto inp;
	}

	printf("Enter granularity (please use a multiple of 4K):    ");
	gets(buffer);
	len = strtoul(buffer, 0, 0);

	if(len % 4096 || addr.LowPart % 4096 || endaddr.LowPart % 4096)
	{
		printf("Well, I said to use a multiple of 4K. Please try again.\n");
	    goto inp;
	}

	C(BestMasterGenPropDefaultSet(cdp->handle));

	printf("Do you want a write_read_compare (1) or only a read (2)? (default: 2): ");
	gets(buffer);
	if(atoi(buffer) == 1)
	{
       doreadwrite = 1;
	   printf("WARNING: This might crash your operating system\n");
	   if(len != 0x1000)
	   {
		   printf("Sorry, I do only take len 0x1000 if you do write read compare.\n");
		   goto inp;
	   }
	}
	else
		doreadwrite = 0;



	printf("mem_read running...\n");
	fprintf(stream, "%s Mem_read_running...\n", asctime (&when));
	((thread_info_t *)tip)->state = RUNNING;


	while (addr.HighPart < endaddr.HighPart || (
		addr.HighPart == endaddr.HighPart && 
		addr.LowPart < endaddr.LowPart))
	{
		if(doreadwrite)
		{
			setup_master_block(cdp->handle, BPPR_DIR_WRITE, addr, 0x10000, len, 
				1, 0, 0,cdp->bitver,WRITE_PPR_REP,0,0);
			setup_master_block(cdp->handle, BPPR_DIR_READ,  addr, 0x14000, len, 
				BPPR_BLK_APPEND, 1, 0x10000,cdp->bitver,WRITE_PPR_REP,1,1);
		}
		else
		{
  			setup_master_block(cdp->handle, BPPR_DIR_READ, addr, 0x0000, 
				len, 1, 0, 0,cdp->bitver,NO_PPR_REP,0,1);
		}
		zw.QuadPart = addr.QuadPart + (LONGLONG) len;
		err2 = BestMasterBlockPageRun(cdp->handle, 1);

		printf("Mem Space: %08x %08x to %08x %08x ",
				addr.HighPart,addr.LowPart,zw.HighPart,zw.LowPart);
		fprintf(stream,"Mem Space: %08x %08x to %08x %08x ",
				addr.HighPart,addr.LowPart,zw.HighPart,zw.LowPart);

		if(err2 == B_E_FUNC)
		{
			status = 0;
			C(BestStatusRegGet(cdp->handle, &status));
			if(status & 0x80)
			{
				printf("MASTER ABORT occured.\n");
				fprintf(stream,"MASTER ABORT occured.\n");
			}
			else
			{
				printf("Functional Onboard Error occured. (Line: %d)\n",__LINE__);
				fprintf(stream,"Functional Onboard Error occured. (Line: %s)\n",__LINE__);
			}
		}
		else
		{
			if(err2 != B_E_OK)
			{
				printf("%s (line %d)\n", BestErrorStringGet(err2), __LINE__);
			}
			else // this is the success case !!!!
			{ // master started without an error
				mycnt = 0;
				while (mycnt++ < 50) // we want to try at the most 100 times that
					// this transfer is finished...
				{
					C(BestStatusRegGet(cdp->handle, &status));
					if(status & STATUS_MASTER_ACTIVE == 0)
						break; // we have a regular stop here
					Sleep(200);
				}
				if(status & STATUS_MASTER_ACTIVE)
				{
					printf("Could not finish memory area.\n");
					fprintf(stream,"Could not finish memory area.\n");
					errflag = 1;
				}
				obserr = getobserror(cdp->handle,cdp->bitver);
				if(obserr)
				{
					printf("Protocol Error: %s.\n",obserr);
					fprintf(stream,"Protocol Error: %s.\n",obserr);
					C(BestObsStatusClear(cdp->handle));
					errflag = 1;
				}
				
				if(!errflag)
				{
					printf("OK.\n");
					fprintf(stream,"OK.\n");
				}
			}
		}
		

//		Sleep(500);
		if (check_card(0, cdp))
		{	break;
		}


		if (((thread_info_t *)tip)->term_request)
		{	break;
		}

		time(&mem_end_run);
		elapsed = difftime(mem_end_run,mem_start_run);

//		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
//		{	break;
//		}
       addr.QuadPart += (LONGLONG) len;
	}

	C(BestMasterStop(cdp->handle));
	cdp->busy = 0;
	((thread_info_t *)tip)->state = TERMINATED;
	printf("mem_read terminating...\n");
	fprintf(stream, "%s Mem_scan terminating...\n", asctime (&when));
	time(&mem_end_run);
	elapsed = difftime(mem_end_run,mem_start_run);
	printf("Elapsed time of memory read test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of memory read test: %.2f minutes.\n", elapsed / 60.0);


	return(0);
}

//---------------------------------------------------------------------------------------------

fct_tab_t fct_tab[] = 
{
	{ "Bus Master Memory Read", mem_read },
	{ "Memory Write/Read/Compare to system memory", mem_write_read_compare },
	{ "CPU access to HPBEST memory space", cpu_mem_test },
	{ "CPU access to HPBEST I/O space", cpu_io_test },
	{ "Peer to peer read/write test", peer2peer },
	{ "PCI Configuration scan", config_scan },
	{ "Stress test", stress_test },
	{ "Memory Scan test", mem_scan },
	{ NULL, NULL }
};

main(int argc, char **argv)
{	b_errtype err;
	unsigned int selection;
	char buf[20];
	int cont = 1;
	unsigned long i;
	int j;
	double elapsed;
	thread_info_t *tip;
	DWORD status;
	card_info_t *cdp;
	struct fct_tab_s *ftp;
	struct tm when;
	time_t now, start_run, end_run;

	system("copy BUS_TEST.LOG BUS_TEST.BAK");
	if ((stream = fopen("BUS_TEST.LOG", "a+")) == NULL)
	{	printf ("Couldn't open log file.\n");
	}

	printf("\nE2926A <IBM CLASSIFIED> (Ver. %s %s)\n", __DATE__, __TIME__);
	fprintf(stream, "\nE2925A/E2926A <IBM CLASSIFIED> (Ver. %s %s)\n", __DATE__, __TIME__);
	printf ("Version 1.00 May 26, 1998\n");
	fprintf (stream, "Version 1.00, May 26, 1998\n");

	time(&now);
	when = *localtime(&now);

	printf("Current time is %s\n", asctime (&when));
	fprintf(stream, "Current time is %s\n",asctime (&when));
	time (&start_run);

	for (j = 1; j < argc; j++)
	{	if (strcmp(argv[j], "-d") == 0)	
		{	dbg = 0; //no output
			printf("No output will be used!!\n");
		}
		if (strcmp(argv[j], "-p") == 0)	
		{	use_ppr = 0;                            // 1=using PPR variations 0=no variations
			write_report = 0;                       // flag for generating a PPR report
			printf("No ppr report will be used or written!!\n");
		}
		if (strcmp(argv[j], "-?") == 0)	
		{	printf("-d \t No Output \n-p \t No PPR reports");
			return (-1);
		}
	}

	printf("The PPR reports are %s\n",write_report==1 ? "On":"Off");
	if (open_all_cards() == 0)
	{	while (cont)
		{	printf("\n-------------------------------------------------------------------------\n");
			for (i = 0, tip = thread_info; i < n_threads; i++, tip++)
			{	GetExitCodeThread(tip->hThread, &status);
				printf("Thread #%d %s  (%s)\n",
					i, status == STILL_ACTIVE ? "active":"terminated", tip->ftp->name);
				fprintf(stream,"Thread #%d %s  (%s)\n",
					i, status == STILL_ACTIVE ? "active":"terminated", tip->ftp->name);
			}
			fprintf(stream, "\n");
			for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
			{	if (cdp->tip)
				{	GetExitCodeThread(cdp->tip->hThread, &status);
					if (status != STILL_ACTIVE)
					{	C(BestMasterStop(cdp->handle));
						cdp->busy = 0;
						cdp->tip = NULL;
					}
				}
				printf("Card #%02d (%s) (bus %d, dev 0x%02lx)  %s",
					i,(cdp->bitver == BEST_32_BIT_CARD ? "E2925A" : "E2926A" ),
					(cdp->devid >> 8) & 0xff, cdp->devid & 0xff, cdp->busy ? "busy":"idle");
				fprintf(stream,"Card #%02d (%s) (bus %d, dev 0x%02lx)  %s",
					i,(cdp->bitver == BEST_32_BIT_CARD ? "E2925A" : "E2926A" ),
					(cdp->devid >> 8) & 0xff, cdp->devid & 0xff, cdp->busy ? "busy":"idle");

				if (cdp->tip)
				{	printf(", used by thread #%d", cdp->tip - thread_info);
					fprintf(stream,", used by thread #%d", cdp->tip - thread_info);
				}
				printf("\n");
				fprintf(stream,"\n");
			}
			printf("\n");
			fprintf(stream,"\n");
			for (i = 1, ftp = fct_tab; ftp->name; i++, ftp++)
			{
				printf("  %d - %s\n", i, ftp->name);
				fprintf(stream,"  %d - %s\n", i, ftp->name);
			}
			printf("  q - Exit\n");
			printf("\nSelect test: ");
			gets(buf);
			selection = atoi(buf);
			printf("\n");
			if (*buf == 'q')
			{	cont = 0;
			}
			else
			{	if (selection >= 1 && selection < i)
				{	spawn(&fct_tab[selection - 1]);
				}
			}
		}
		printf("waiting for %d threads to terminate...\n", n_threads);
		fprintf(stream,"Waiting for %d threads to terminate...\n", n_threads);
		for (i = 0; i < n_threads; i++)
		{	thread_info[i].term_request = 1;
		}

		Sleep(2000);   // allow some time for the threads to terminate
		for (i = 0, tip = thread_info; i < n_threads; i++, tip++)
		{	GetExitCodeThread(tip->hThread, &status);
			if (status == STILL_ACTIVE)
			{	printf("killing thread #%d\n", i);
				TerminateThread(tip->hThread, 0);
				fprintf(stream,"Killing thread #%d\n", i);
			}
		}
		close_all_cards();
	}

	else 
	{	
		for (i = 1, ftp = fct_tab; ftp->name; i++, ftp++)
			{
				printf("  %d - %s\n", i, ftp->name);
				fprintf(stream,"  %d - %s\n", i, ftp->name);
			}
		printf("Hit Enter to Exit!\n");
		getch();
		return (-1);
	}
	printf("press RETURN to continue");
	gets(buf);

	time(&now);
	when = *localtime(&now);
	time(&end_run);
	printf("Ending time is %s\n", asctime (&when));
	fprintf(stream, "Ending time is %s\n",asctime (&when));

	elapsed = difftime(end_run,start_run);
	printf("Elapsed time of test(s): %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of tests(s): %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"------------------------------------------------------------------------------\n\n\n");
	
	fclose(stream);
	return(0);
}
